-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
--
-- DO NOT CREATE functions, that take filenames as parameters !!!
--
--   * every function is callable by every user that can connect to the db
--   * which means every functions needs to be safe to be called
--   * having a function that takes a filename as parameter is a severe security flaw
--   * it would allow every user to actually write verywhere to the disk of the pg-server
--
-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
-- lots of interdependencies, so lots of cascade to clear  out everything
--------------------------------------------------------------------------------
-- DROP TABLE IF EXISTS tlog.dblogfiles_imported CASCADE;
-- DROP TABLE IF EXISTS tlog.dblog CASCADE;
-- DROP TYPE IF EXISTS tlog.dblog_loglevel CASCADE;
-- DROP TYPE IF EXISTS tlog.dblog_logtype CASCADE;
-- DROP TYPE IF EXISTS tlog.dblog_logsource CASCADE;
-- DROP SEQUENCE IF EXISTS tlog.dblog_id CASCADE;

-- DROP FUNCTION IF EXISTS tlog.txid_ifassigned;
-- DROP FUNCTION IF EXISTS tlog.in_transaction;

-- DROP FUNCTION IF EXISTS tlog.dblog__raisenotice;
-- DROP FUNCTION IF EXISTS tlog.dblog__Logfile__full_path__Get;
-- DROP FUNCTION IF EXISTS tlog.dblog__Logfile__exists;
-- DROP FUNCTION IF EXISTS tlog.dblog__Logfile__delete;
-- DROP FUNCTION IF EXISTS tlog.dblog__Logfiles__delete;
-- DROP FUNCTION IF EXISTS tlog.dblog__Logfile_content__cleanup;
-- DROP FUNCTION IF EXISTS tlog.dblog__create__temp_tables__from__Logfile;
-- DROP FUNCTION IF EXISTS tlog.dblog__into__from__Logfiles;
-- DROP FUNCTION IF EXISTS tlog.dblog__Logfile__write_line;
-- DROP FUNCTION IF EXISTS dblog.in_transactiondblog.log(tlog.dblog_logtype, tlog.dblog_loglevel, tlog.dblog_logsource, varchar, inet, integer, varchar, bigint, varchar, varchar);
-- DROP FUNCTION IF EXISTS dblog.in_transaction(tlog.dblog_logtype, tlog.dblog_loglevel, tlog.dblog_logsource, bigint, varchar, varchar);


--------------------------------------------------------------------------------
-- helper for python functions
CREATE OR REPLACE FUNCTION tlog.dblog__raisenotice(a_message varchar) RETURNS void AS $$
BEGIN
  RAISE NOTICE '%', a_message;
END $$ LANGUAGE plpgsql;


--------------------------------------------------------------------------------
-- create the filename in python and with parameters
-- this way no function takes a filename as parameter
-- and thus is safer (we DO WRITE to filesystem here, within the db-server ...)
-- configurate access right in a way that only the dblog schema functions can execute this one??

-- gets dir of logfile
CREATE OR REPLACE FUNCTION tlog.dblog__Logfile__dir_name__Get() RETURNS varchar AS $$
DECLARE
  _DataDir varchar;
  _OS varchar;
  _LogOn boolean;
  _LogSubDir varchar;
  _LogDir varchar;

BEGIN
  SELECT
    current_setting('data_directory'),
    current_setting('dynamic_shared_memory_type'),
    current_setting('logging_collector'),
    COALESCE(current_setting('log_directory'), '')
  INTO
    _DataDir,
    _OS,
    _LogON,
    _LogSubDir
  ;

  _LogDir = _DataDir;
  IF (_OS = 'windows') THEN
    IF (_LogON) THEN
      IF (NOT (_LogSubDir = '')) THEN
        _LogDir = CONCAT(_LogDir, '/', _LogSubDir);
      END IF;
    END IF;
  END IF;

  RETURN _LogDir;
END;
$$ LANGUAGE plpgsql STABLE;


-- gets full path including filenam of logfile
CREATE OR REPLACE FUNCTION tlog.dblog__Logfile__full_path__Get(a_date date DEFAULT today(), a_pid integer DEFAULT pg_backend_pid(), a_database varchar default current_database()) RETURNS varchar AS $$
DECLARE
  _LogDir varchar;

BEGIN
  SELECT
    tlog.dblog__Logfile__dir_name__Get()
  INTO
    _LogDir
  ;

  RETURN CONCAT(_LogDir, '/csvlog.', a_database, '.', to_char(a_date, 'YYYYMMDD'), '.', a_pid::varchar, '.csv');
END;
$$ LANGUAGE plpgsql STABLE;

--------------------------------------------------------------------------------
-- delete csvlogfiles from disk, which have an entry in tlog.dblogfiles_imported
CREATE OR REPLACE FUNCTION tlog.dblog__Logfiles__delete() RETURNS void AS $$
BEGIN
  PERFORM tlog.dblog__Logfile__delete(dblfi_date, dblfi_pid, dblfi_database) FROM tlog.dblogfiles_imported;

  RETURN;
END;
$$ LANGUAGE plpgsql;
--------------------------------------------------------------------------------
-- delete entries in tltlog.dblogfiles_imported, which do not have a corresponding csvlogfile
CREATE OR REPLACE FUNCTION tlog.dblog__dblogfiles_imported__cleanup() RETURNS void AS $$
BEGIN
  DELETE FROM
    tlog.dblogfiles_imported
  WHERE
    NOT(tlog.dblog__Logfile__exists(dblfi_date, dblfi_pid, dblfi_database))
  ;

  RETURN;
END;
$$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- manually import a single csvlogfile into tlog.dblog
--   1) check dblog.csvlogfiles for entries which exist no longer and delete from dblog.csvlogfiles
--   2) check dblog.csvlogfiles for files which are still on disk and delete those file
--   3) import file + add to dblog.csvlogfiles
CREATE OR REPLACE FUNCTION tlog.dblog__Logfile__import(a_date date DEFAULT today(), a_pid integer DEFAULT pg_backend_pid(), a_database varchar default current_database()) RETURNS void AS $$
DECLARE
  _logfilename varchar;
  _temptable varchar;
  _SQL varchar;

BEGIN
  IF (a_database = current_database()) THEN
    PERFORM pg_advisory_xact_lock(-1, -1);

    -- delete entries in tltlog.dblogfiles_imported, which do not have a corresponding csvlogfile
    PERFORM tlog.dblog__dblogfiles_imported__cleanup();
    -- delete csvlogfiles from disk, which have an entry in tltlog.dblogfiles_imported
    PERFORM tlog.dblog__Logfiles__delete();

    IF ( tlog.dblog__Logfile__exists(a_date, a_pid, a_database) ) THEN
      SELECT
        tlog.dblog__Logfile__full_path__Get(a_date, a_pid, a_database)
      INTO
        _logfilename
      ;

      _temptable = FORMAT('dblog_tmp_%s_%s', to_char(today(), 'YYYYMMDD'), pg_backend_pid()::varchar);
      EXECUTE FORMAT('DROP TABLE IF EXISTS %s', _temptable);
      EXECUTE FORMAT('CREATE TEMP TABLE %s(dbl_id bigint, logentry json)', _temptable);

      -- QUOTE e'\x01' DELIMITER e'\x02
      -- This is needed, to allow json-escaped characters inside a json-string
      -- linebreaks are escaped in json-strings with \n
      _SQL = FORMAT('COPY %s FROM ''%s'' CSV QUOTE e''\x01'' DELIMITER e''\x02'';', _temptable, _logfilename);
      -- RAISE NOTICE 'Import: %', _SQL;
      EXECUTE _SQL;

      _SQL = FORMAT(
        $SQL$
          WITH src AS
          (
            SELECT
              jsonrow.*
            FROM
              %s AS tmptable
              CROSS JOIN json_populate_record(NULL::tlog.dblog, tmptable.logentry) AS jsonrow
          )
          INSERT INTO
            tlog.dblog
          SELECT
            *
          FROM
            src
          WHERE
            NOT EXISTS(SELECT dbl_id FROM tlog.dblog WHERE dbl_id = src.dbl_id)
          ;
        $SQL$, _temptable)
      ;
      -- RAISE NOTICE 'IMPORT: %', _SQL;
      EXECUTE _SQL;

      INSERT INTO
        tlog.dblogfiles_imported (dblfi_date, dblfi_pid, dblfi_database)
      VALUES
        (a_date, a_pid, a_database)
      ON CONFLICT DO NOTHING
      ;

      RAISE NOTICE 'tlog.dblog__Logfile__import ok: %', _temptable;
    ELSE
      RAISE NOTICE 'tlog.dblog__Logfile__import: does not exists or already imported %, %, %', to_char(a_date, 'YYYYMMDD'), a_pid, a_database;
    END IF;
  ELSE
    RAISE WARNING 'tlog.dblog__Logfile__import: wrong database %, %, %', to_char(a_date, 'YYYYMMDD'), a_pid, a_database;
  END IF;
END;
$$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- build the csv-line for the logfile and call tlog.dblog__Logfile__write_line
CREATE OR REPLACE FUNCTION tlog.dblog__log_line__create(
    a_logtype tlog.dblog_logtype,
    a_loglevel tlog.dblog_loglevel,
    a_logsource tlog.dblog_logsource,
    a_conn_user varchar,
    a_conn_ip inet,
    a_conn_pid integer,
    a_conn_application varchar,
    a_parent_id bigint,
    a_ctx varchar,
    a_msg varchar
) RETURNS bigint AS $$
DECLARE
  _logid bigint;
  _logrow tlog.dblog%rowtype;
  _json json;
  _csv varchar;

BEGIN
   _logid := nextval('tlog.dblog_id'::regclass);

  SELECT
    _logid,
    timezone('utc'::text, clock_timestamp()),
    timezone('utc'::text, current_timestamp), -- current_timestamp => start time of the current transaction
    a_logtype,
    a_loglevel,
    a_logsource,
    a_conn_user,
    a_conn_ip,
    a_conn_pid,
    a_conn_application,
    pg_current_xact_id_if_assigned()::varchar, -- tlog.txid_ifassigned(),
    pg_current_snapshot(),
    a_parent_id,
    a_ctx,
    a_msg
  INTO
    _logrow
  ;

  _json = row_to_json(_logrow);
  -- see: tlog.dblog__Logfile__import
  _csv = CONCAT(_logrow.dbl_id::varchar, e'\x02', e'\x01', _json::varchar, e'\x01');
  -- RAISE NOTICE '%', _csv;

  PERFORM tlog.dblog__Logfile__log_line__write(_csv);

  RETURN _logid;
END $$ LANGUAGE plpgsql;
--------------------------------------------------------------------------------
CREATE OR REPLACE FUNCTION tlog.dblog__log_line__create(
    a_logtype tlog.dblog_logtype,
    a_loglevel tlog.dblog_loglevel,
    a_logsource tlog.dblog_logsource,
    a_parent_id bigint,
    a_ctx varchar,
    a_msg varchar
  ) 
  RETURNS bigint 
  AS $$
  BEGIN
    RETURN
    (
      SELECT
        tlog.dblog__log_line__create
        (
          a_logtype, a_loglevel, a_logsource,
          session_user::varchar, inet_client_addr(), pg_backend_pid(), current_setting('application_name'), -- session_user without brackets !!!
          a_parent_id, a_ctx, a_msg
        )
      )
    ;
  END $$ LANGUAGE plpgsql;
---

/*
CREATE OR REPLACE FUNCTION tlog.in_transaction() RETURNS boolean AS $$
  SELECT transaction_timestamp() != statement_timestamp();
$$ LANGUAGE SQL;

CREATE OR REPLACE FUNCTION tlog.txid_ifassigned() RETURNS varchar 
  AS $$
    SELECT
      CASE WHEN
        tlog.in_transaction() THEN
          pg_current_xact_id_if_assigned()::varchar
      ELSE
        NULL::varchar
      END;
  $$ LANGUAGE SQL;
*/  
---
--------------------------------------------------------------------------------
/*
CREATE OR REPLACE FUNCTION tlog.dblog__Logfile__exists(a_date date DEFAULT today(), a_pid integer DEFAULT pg_backend_pid(), a_database varchar default current_database()) RETURNS boolean AS
$$
  import os

  _plan = plpy.prepare("SELECT tlog.dblog__Logfile__full_path__Get($1::date, $2::integer, $3::varchar) as filename", ["date","integer","varchar"])
  _ret = plpy.execute(_plan, [a_date, a_pid, a_database])
  _filename = _ret[0]["filename"]

  if os.path.exists(_filename):
    return True
  else:
    return False
$$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- delete a csvlogfile from disk
CREATE OR REPLACE FUNCTION tlog.dblog__Logfile__delete(a_date date DEFAULT today(), a_pid integer DEFAULT pg_backend_pid(), a_database varchar default current_database()) RETURNS void AS
$$
  import os

  _plan = plpy.prepare("SELECT tlog.dblog__Logfile__full_path__Get($1::date, $2::integer, $3::varchar) as filename", ["date","integer","varchar"])
  _ret = plpy.execute(_plan, [a_date, a_pid, a_database])
  _filename = _ret[0]["filename"]

  if os.path.exists(_filename):
    os.remove(_filename)
$$ LANGUAGE plpython3u;


--------------------------------------------------------------------------------
-- manually import csvlogfiles into tlog.dblog
CREATE OR REPLACE FUNCTION tlog.dblog__Logfiles__import(currentsession boolean, othersessions boolean, thisday boolean, olderthanthisday boolean) RETURNS void AS
$$
  import glob
  import datetime

  _plan = plpy.prepare("SELECT tlog.dblog__Logfile__import($1::date, $2::integer, $3::varchar) as filename", ["date","integer","varchar"])
  _ret = plpy.execute("SELECT pg_backend_pid() as pid, current_database() as db, tlog.dblog__Logfile__dir_name__Get() as logdir")

  _current_pid = _ret[0]["pid"]
  _current_db = _ret[0]["db"]
  _LogDir = _ret[0]["logdir"] + '/'
  _LenDir = len(_LogDir)

  for _file in glob.glob(_LogDir + "csvlog.*.*.*.csv"):
    _filename = _file[_LenDir:]
    _parts = _filename.split(".")
    _partscount = len(_parts)
    _plan_notice = plpy.prepare("SELECT ($1::varchar);", ["varchar"])
    _message = "tlog.dblog__Logfiles__import: " + _filename
    plpy.execute(_plan_notice, [_message])
    if ((_partscount == 5) and (_parts[0] == 'csvlog') and (_parts[4] == 'csv')):
      _database = _parts[1]
      try:
        _pid = int(_parts[3])
      except Exception as _E:
        _pid = 0

      try:
        _date = datetime.datetime.strptime(_parts[2], '%Y%m%d')
      except Exception as _E:
        _pid = 0

      _doimport = False

      if (currentsession and (_pid == _current_pid)):
        _doimport = True
      if (othersessions and (_pid != _current_pid)):
        _doimport = True

      _diff = datetime.datetime.now() - _date
      if (thisday and (_diff.days == 0)):
        _doimport = True
      if (olderthanthisday and (_diff.days > 0)):
        _doimport = True

      if ((_pid > 0) and _doimport and (_database == _current_db)):
        _ret = plpy.execute(_plan, [_date, _pid, _database])
$$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- returns the existing importlogfiles without showing actual filenames
CREATE OR REPLACE FUNCTION tlog.dblog__Logfiles__import__log_line__create() RETURNS SETOF tlog.dblogfiles_imported AS
$$
  import glob
  import datetime

  _ret = plpy.execute("SELECT pg_backend_pid() as pid, current_database() as db, tlog.dblog__Logfile__dir_name__Get() as logdir")

  _current_pid = _ret[0]["pid"]
  _current_db = _ret[0]["db"]
  _LogDir = _ret[0]["logdir"] + '/'
  _LenDir = len(_LogDir)

  for _file in glob.glob(_LogDir + "csvlog.*.*.*.csv"):
    _filename = _file[_LenDir:]
    _parts = _filename.split(".")
    _partscount = len(_parts)
    _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])
    _message = "tlog.dblog__Logfiles__import__log_line__create: " + _filename
    plpy.execute(_plan_notice, [_message])
    if ((_partscount == 5) and (_parts[0] == 'csvlog') and (_parts[4] == 'csv')):
      _database = _parts[1]
      try:
        _pid = int(_parts[3])
      except Exception as _E:
        _pid = 0

      try:
        _date = datetime.datetime.strptime(_parts[2], '%Y%m%d')
      except Exception as _E:
        _pid = 0

      if ((_pid > 0)):
        yield(_date, _database, _pid)
$$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- write a line to the csvlogfile
--   the encoding of the file-open needs to be the same as the default db-encoding
--   this should be always utf-8, considering the stand of things at moment of coding
CREATE OR REPLACE FUNCTION tlog.dblog__Logfile__log_line__write(csvline varchar) RETURNS void AS
$$
  import os

  _ret = plpy.execute("SELECT tlog.dblog__Logfile__full_path__Get() as filename")
  _logfilename = _ret[0]["filename"]

  with open(_logfilename, 'a', encoding='utf8') as fd:
    fd.write(csvline + f'\n')
$$ LANGUAGE plpython3u;
---
*/
